import json
import win32con
import win32api
import win32gui
import time
from ctypes import POINTER, windll, Structure, cast, CFUNCTYPE, c_int, c_uint, c_void_p, c_bool
from comtypes import GUID
from ctypes.wintypes import HANDLE, DWORD
import mqttClient
import logging
import mylog

mylog.setup_logger()


# 显示状态和电源状态
screenStat = {
    'display': None,    #None,on,off
    'acdcpower': None,  #None,ac,dc
    'battery': None,
    'time': time.strftime('%Y-%m-%d %H:%M:%S', time.localtime(time.time()))
}
mqttclient = mqttClient.conn_mqtt()
mqttclient.loop_start()

PBT_POWERSETTINGCHANGE = 0x8013
GUID_CONSOLE_DISPLAY_STATE = '{6FE69556-704A-47A0-8F24-C28D936FDA47}'
GUID_ACDC_POWER_SOURCE = '{5D3E9A59-E9D5-4B00-A6BD-FF34FF516548}'
GUID_BATTERY_PERCENTAGE_REMAINING = '{A7AD8041-B45A-4CAE-87A3-EECBB468A9E1}'
GUID_MONITOR_POWER_ON = '{02731015-4510-4526-99E6-E5A17EBD1AEA}'
GUID_SYSTEM_AWAYMODE = '{98A7F580-01F7-48AA-9C0F-44352C29E5C0}'


class POWERBROADCAST_SETTING(Structure):
    _fields_ = [("PowerSetting", GUID),
                ("DataLength", DWORD),
                ("Data", DWORD)]


def wndproc(hwnd, msg, wparam, lparam):
    '''
    windwos电源事件回调处理函数
    事件详细定义参考  https://learn.microsoft.com/zh-cn/windows/win32/power/power-setting-guids
    :param hwnd:
    :param msg:
    :param wparam:
    :param lparam:
    :return:
    '''
    if msg == win32con.WM_POWERBROADCAST:
        if wparam == win32con.PBT_APMPOWERSTATUSCHANGE:
            logging.info('Power status has changed')
        if wparam == win32con.PBT_APMRESUMEAUTOMATIC:
            logging.info('System resume')
        if wparam == win32con.PBT_APMRESUMESUSPEND:
            logging.info('System resume by user input')
        if wparam == win32con.PBT_APMSUSPEND:
            logging.info('System suspend')
        if wparam == PBT_POWERSETTINGCHANGE:
            logging.info('Power setting changed...')
            settings = cast(lparam, POINTER(POWERBROADCAST_SETTING)).contents
            power_setting = str(settings.PowerSetting)
            data_length = settings.DataLength
            data = settings.Data
            if power_setting == GUID_CONSOLE_DISPLAY_STATE:
                if data == 0:
                    logging.info('Display off')
                    screenStat['display'] = 'off'
                    screenStat['time'] = time.strftime('%Y-%m-%d %H:%M:%S', time.localtime(time.time()))
                if data == 1:
                    logging.info('Display on')
                    screenStat['display'] = 'on'
                    screenStat['time'] = time.strftime('%Y-%m-%d %H:%M:%S', time.localtime(time.time()))
                if data == 2: logging.info('Display dimmed')
            elif power_setting == GUID_ACDC_POWER_SOURCE:
                if data == 0:
                    logging.info('AC power')
                    screenStat['acdcpower'] = 'ac'
                    screenStat['time'] = time.strftime('%Y-%m-%d %H:%M:%S', time.localtime(time.time()))
                if data == 1:
                    logging.info('Battery power')
                    screenStat['acdcpower'] = 'dc'
                    screenStat['time'] = time.strftime('%Y-%m-%d %H:%M:%S', time.localtime(time.time()))
                if data == 2: logging.info('Short term power')
            elif power_setting == GUID_BATTERY_PERCENTAGE_REMAINING:
                logging.info('battery remaining: %s' % data)
                screenStat['battery'] = str(data)
                screenStat['time'] = time.strftime('%Y-%m-%d %H:%M:%S', time.localtime(time.time()))
            elif power_setting == GUID_MONITOR_POWER_ON:
                if data == 0:
                    logging.info('Monitor off')
                    screenStat['display'] = 'off'
                    screenStat['time'] = time.strftime('%Y-%m-%d %H:%M:%S', time.localtime(time.time()))
                if data == 1:
                    logging.info('Monitor on')
                    screenStat['display'] = 'on'
                    screenStat['time'] = time.strftime('%Y-%m-%d %H:%M:%S', time.localtime(time.time()))
            elif power_setting == GUID_SYSTEM_AWAYMODE:
                if data == 0: logging.info('Exiting away mode')
                if data == 1: logging.info('Entering away mode')
            else:
                logging.info('unknown GUID')
            mqttClient.publish(mqttclient, json.dumps(screenStat))
        return True
    return False


if __name__ == "__main__":
    logging.info("*** STARTING ***")
    hinst = win32api.GetModuleHandle(None)
    wndclass = win32gui.WNDCLASS()
    wndclass.hInstance = hinst
    wndclass.lpszClassName = "testWindowClass"
    CMPFUNC = CFUNCTYPE(c_bool, c_int, c_uint, c_uint, c_void_p)
    wndproc_pointer = CMPFUNC(wndproc)
    wndclass.lpfnWndProc = {win32con.WM_POWERBROADCAST: wndproc_pointer}
    try:
        myWindowClass = win32gui.RegisterClass(wndclass)
        hwnd = win32gui.CreateWindowEx(win32con.WS_EX_LEFT,
                                       myWindowClass,
                                       "testMsgWindow",
                                       0,
                                       0,
                                       0,
                                       win32con.CW_USEDEFAULT,
                                       win32con.CW_USEDEFAULT,
                                       0,
                                       0,
                                       hinst,
                                       None)
    except Exception as e:
        logging.info("Exception: %s" % str(e))

    if hwnd is None:
        logging.info("hwnd is none!")
    else:
        logging.info("hwnd: %s" % hwnd)

    guids_info = {
        'GUID_MONITOR_POWER_ON' : GUID_MONITOR_POWER_ON,
        'GUID_SYSTEM_AWAYMODE' : GUID_SYSTEM_AWAYMODE,
        'GUID_CONSOLE_DISPLAY_STATE' : GUID_CONSOLE_DISPLAY_STATE,
        'GUID_ACDC_POWER_SOURCE' : GUID_ACDC_POWER_SOURCE,
        'GUID_BATTERY_PERCENTAGE_REMAINING' : GUID_BATTERY_PERCENTAGE_REMAINING
    }
    for name, guid_info in guids_info.items():
        result = windll.user32.RegisterPowerSettingNotification(HANDLE(hwnd), GUID(guid_info), DWORD(0))
        logging.info('registering %s', name)
        logging.info('result:%s', hex(result))
        logging.info('lastError:%s', win32api.GetLastError())

    logging.info('\n------------------Entering loop----------------')
    while True:
        win32gui.PumpWaitingMessages()
        time.sleep(1)
